<?php
if (!defined('ABSPATH')) exit;

class SW_Strava_Shortcodes {

  public function init(): void {
    add_shortcode('strava_activities', [$this, 'sc_activities']);
    add_shortcode('strava_stats', [$this, 'sc_stats']);
    add_shortcode('strava_latest', [$this, 'sc_latest']);
    add_shortcode('strava_photos', [$this, 'sc_photos']);
    add_shortcode('strava_top_activity', [$this, 'sc_top_activity']);

    add_action('wp_ajax_sw_strava_activities_page', [$this, 'ajax_activities_page']);
    add_action('wp_ajax_nopriv_sw_strava_activities_page', [$this, 'ajax_activities_page']);
  }

  private function parse_types_csv(string $type_spec): array {
    $type_spec = trim($type_spec);
    if ($type_spec === '' || strtolower($type_spec) === 'all') return ['all'];

    $parts = array_values(array_filter(array_map('trim', explode(',', $type_spec))));
    $parts = array_map(function($x){ return strtolower($x); }, $parts);

    return !empty($parts) ? $parts : ['all'];
  }

  private function normalize_title_template($raw): string {
    // Keep placeholders like %date intact. sanitize_text_field() would strip %da as a hex-octet.
    $raw = is_string($raw) ? $raw : '';
    $raw = wp_unslash($raw);
    return trim(wp_strip_all_tags($raw));
  }

  private function is_title_none(string $template): bool {
    return strtolower(trim($template)) === 'none';
  }


  /**
   * Activities table (AJAX paginated)
   */
  public function sc_activities($atts): string {
    wp_enqueue_style('sw-strava-frontend');
    wp_enqueue_script('sw-strava-frontend');

    $atts = shortcode_atts([
      'year' => gmdate('Y'),
      'type' => 'all',
      'per_page' => 20,
      'page' => 1,
      'fields' => 'name,type,distance,total_elevation_gain,start_date',
      'title' => '',
    ], $atts);

    $api = new SW_Strava_API();

    $year_spec = sanitize_text_field($atts['year']);
    $years = $api->resolve_years($year_spec);

    $dateText = $this->format_stats_year_title($year_spec, $years);
    $title_template = $this->normalize_title_template($atts['title'] ?? '');
    $hide_title = $this->is_title_none($title_template);
    $title_template = $this->normalize_title_template($atts['title'] ?? '');
    $hide_title = $this->is_title_none($title_template);
    $fallbackTitle = 'Aktivitäten ' . $dateText;
    $title_template = $this->normalize_title_template($atts['title'] ?? '');
    $hide_title = $this->is_title_none($title_template);
    $title = $hide_title ? '' : $this->build_title($title_template, $fallbackTitle, $dateText);

    $type_spec = sanitize_text_field($atts['type']);
    $types = $this->parse_types_csv($type_spec);

    $per_page = max(1, min(100, intval($atts['per_page'])));
    $page = max(1, intval($atts['page']));

    $fields = array_values(array_filter(array_map('trim', explode(',', (string)$atts['fields']))));
    if (empty($fields)) $fields = ['name','type','distance','total_elevation_gain','start_date'];

    $resp = $api->cached_activities_for_years($years);
    if (!$resp['ok']) return '<div class="sw-strava-error">' . esc_html($resp['error']) . '</div>';

    $acts = $resp['data'];

    if ($types !== ['all']) {
      $acts = array_values(array_filter($acts, function ($a) use ($types) {
        $t = strtolower((string)($a['type'] ?? ''));
        return $t !== '' && in_array($t, $types, true);
      }));
    }

    $total = count($acts);
    $total_pages = max(1, (int)ceil($total / $per_page));
    if ($page > $total_pages) $page = $total_pages;

    $offset = ($page - 1) * $per_page;
    $slice = array_slice($acts, $offset, $per_page);

    $dateText = $this->format_stats_year_title($year_spec, $years);
    $fallbackTitle = 'Aktivitäten ' . $dateText;
    $title_template = $this->normalize_title_template($atts['title'] ?? '');
    $hide_title = $this->is_title_none($title_template);
    $title = $hide_title ? '' : $this->build_title($title_template, $fallbackTitle, $dateText);

    ob_start(); ?>
    <div class="sw-strava sw-strava-table"<?php echo $this->design_style_attr(); ?>
         data-year="<?php echo esc_attr($year_spec); ?>"
         data-type="<?php echo esc_attr($type_spec); ?>"
         data-per-page="<?php echo esc_attr($per_page); ?>"
         data-page="<?php echo esc_attr($page); ?>"
         data-sort-field="start_date"
         data-sort-dir="desc"
         data-fields="<?php echo esc_attr(implode(',', $fields)); ?>">

      <?php if (!$hide_title): ?>
        <div class="sw-title"><?php echo esc_html($title); ?></div>
      <?php endif; ?>

      <table>
        <thead>
          <tr>
            <th data-sort="name">Aktivität</th>
            <th data-sort="type">Typ</th>
            <th data-sort="distance">Distanz (km)</th>
            <th data-sort="total_elevation_gain">Höhenmeter (m)</th>
            <th data-sort="start_date">Datum</th>
          </tr>
        </thead>
        <tbody class="sw-strava-rows">
          <?php foreach ($slice as $a):
            $name = $a['name'] ?? '';
            $atype = $a['type'] ?? '';
            $km = floatval($a['distance'] ?? 0) / 1000;
            $elev = floatval($a['total_elevation_gain'] ?? 0);
            $dt = $a['start_date_local'] ?? ($a['start_date'] ?? '');
            $dt_fmt = $dt ? date_i18n('d.m.Y H:i', strtotime($dt)) : '–';
            $id = $a['id'] ?? '';
            $url = $id ? ('https://www.strava.com/activities/' . rawurlencode((string)$id)) : '';
          ?>
          <tr>
            <td>
              <?php if ($url): ?>
                <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html($name); ?></a>
              <?php else: ?>
                <?php echo esc_html($name); ?>
              <?php endif; ?>
            </td>
            <td><?php echo $this->type_with_icon($atype); ?></td>
            <td><?php echo $this->dash_if_zero($km, 2); ?></td>
            <td><?php echo $this->dash_if_zero_int($elev); ?></td>
            <td><?php echo esc_html($dt_fmt); ?></td>
          </tr>
          <?php endforeach; ?>
        </tbody>
      </table>

      <?php if ($total_pages > 1): ?>
        <div class="sw-strava-pager">
          <button class="sw-strava-prev" type="button" <?php disabled($page <= 1); ?>>Prev</button>
          <span class="sw-strava-pageinfo">
            Seite <strong class="sw-strava-page"><?php echo esc_html($page); ?></strong> /
            <strong class="sw-strava-pages"><?php echo esc_html($total_pages); ?></strong>
            <span class="sw-strava-total">(<?php echo esc_html($total); ?>)</span>
          </span>
          <button class="sw-strava-next" type="button" <?php disabled($page >= $total_pages); ?>>Next</button>
          <span class="sw-strava-loading" style="display:none;">Lade…</span>
        </div>
      <?php endif; ?>
    </div>
    <?php
    return ob_get_clean();
  }

  /**
   * Latest activity card (clickable)
   */
  public function sc_latest($atts): string {
    wp_enqueue_style('sw-strava-frontend');

    $atts = shortcode_atts([
      'year' => 'current',
      'title' => '',
    ], $atts);

    $api = new SW_Strava_API();
    $year_spec = sanitize_text_field($atts['year']);
    $years = $api->resolve_years($year_spec);

    $resp = $api->cached_activities_for_years($years);
    if (!$resp['ok']) return '<div class="sw-strava-error">' . esc_html($resp['error']) . '</div>';

    $acts = $resp['data'];
    if (empty($acts)) return '<div class="sw-strava">Keine Aktivitäten gefunden.</div>';

    $a = $acts[0];

    $name = $a['name'] ?? '';
    $atype = $a['type'] ?? '';
    $km = floatval($a['distance'] ?? 0) / 1000;
    $elev = floatval($a['total_elevation_gain'] ?? 0);
    $dt = $a['start_date_local'] ?? ($a['start_date'] ?? '');
    $dt_fmt = $dt ? date_i18n('d.m.Y H:i', strtotime($dt)) : '–';

    $id = $a['id'] ?? '';
    $url = $id ? ('https://www.strava.com/activities/' . rawurlencode((string)$id)) : '';

    $dateText = $this->format_stats_year_title($year_spec, $years);
    $fallbackTitle = 'Letzte Aktivität ' . $dateText;
    $title_template = $this->normalize_title_template($atts['title'] ?? '');
    $hide_title = $this->is_title_none($title_template);
    $title = $hide_title ? '' : $this->build_title($title_template, $fallbackTitle, $dateText);

    $has_custom_title = (!$hide_title && $title_template !== '');

    ob_start(); ?>
    <div class="sw-strava sw-strava-latest"<?php echo $this->design_style_attr(); ?>>
      <div class="sw-card">
        <?php if ($has_custom_title): ?>
          <div class="sw-title"><?php echo esc_html($title); ?></div>
          <div class="sw-top-name" style="margin-bottom:6px;">
            <?php if ($url): ?>
              <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html($name); ?></a>
            <?php else: ?>
              <?php echo esc_html($name); ?>
            <?php endif; ?>
          </div>
        <?php elseif (!$hide_title): ?>
          <div class="sw-title">
            <?php if ($url): ?>
              <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html($name); ?></a>
            <?php else: ?>
              <?php echo esc_html($name); ?>
            <?php endif; ?>
          </div>
        <?php endif; ?>
        <div class="sw-meta">
          <span><?php echo $this->type_with_icon($atype); ?></span>
          <span><?php echo $this->dash_if_zero($km, 2); ?> km</span>
          <span><?php echo $this->dash_if_zero_int($elev); ?> hm</span>
          <span><?php echo esc_html($dt_fmt); ?></span>
        </div>
      </div>
    </div>
    <?php
    return ob_get_clean();
  }

  /**
   * Stats table
   */
  public function sc_stats($atts): string {
    wp_enqueue_style('sw-strava-frontend');

    $atts = shortcode_atts([
      'year' => 'current',
      'group' => 'total', // total|type
      'sort' => 'count desc, km desc',
      'type' => 'all', // all|Ride,Run,Swim
      'title' => '',
    ], $atts);

    $api = new SW_Strava_API();

    $year_spec = sanitize_text_field($atts['year']);
    $years = $api->resolve_years($year_spec);
    $group = sanitize_text_field($atts['group']);

    $type_filter = sanitize_text_field($atts['type'] ?? 'all');
    $type_filter_list = [];
    if ($type_filter && strtolower($type_filter) !== 'all') {
      $type_filter_list = array_values(array_filter(array_map('trim', explode(',', $type_filter))));
      $type_filter_list = array_map('strtolower', $type_filter_list);
    }

    $acts_resp = $api->cached_activities_for_years($years);
    if (!$acts_resp['ok']) return '<div class="sw-strava-error">' . esc_html($acts_resp['error']) . '</div>';

    $stats = $api->compute_stats_from_activities($acts_resp['data'], 0);

    $dateText = $this->format_stats_year_title($year_spec, $years);
    $fallbackTitle = 'Statistiken ' . $dateText;
    $title_template = $this->normalize_title_template($atts['title'] ?? '');
    $hide_title = $this->is_title_none($title_template);
    $title = $hide_title ? '' : $this->build_title($title_template, $fallbackTitle, $dateText);

    ob_start(); ?>
    <div class="sw-strava sw-strava-stats"<?php echo $this->design_style_attr(); ?>>
      <div class="sw-card">
        <?php if (!$hide_title): ?>
          <div class="sw-title"><?php echo esc_html($title); ?></div>
        <?php endif; ?>

        <div class="sw-kpis">
          <div><strong><?php echo $this->dash_if_zero($stats['total_km'] ?? 0, 2); ?></strong><span> km gesamt</span></div>
          <div><strong><?php echo $this->dash_if_zero_int($stats['total_elev_m'] ?? 0); ?></strong><span> hm gesamt</span></div>
        </div>

        <?php if ($group === 'type'): ?>
          <h4>Nach Sportart</h4>
          <table>
            <thead>
              <tr><th>Typ</th><th>Anzahl</th><th>km</th><th>hm</th><th>Längste</th></tr>
            </thead>
            <tbody>
              <?php
              $rows = [];
              foreach (($stats['by_type'] ?? []) as $type => $t) {
                $long = $stats['longest_by_type'][$type] ?? null;
                $rows[] = [
                  'type' => $type,
                  'count' => intval($t['count'] ?? 0),
                  'km' => round(floatval($t['distance_m'] ?? 0) / 1000, 2),
                  'hm' => round(floatval($t['elev_m'] ?? 0), 0),
                  'longest_km' => $long ? round(floatval($long['distance_m'] ?? 0) / 1000, 2) : 0,
                  'longest_name' => $long['name'] ?? '',
                  'longest_id' => $long['id'] ?? '',
                ];
              }

              
              if (!empty($type_filter_list)) {
                $rows = array_values(array_filter($rows, function($r) use ($type_filter_list) {
                  return in_array(strtolower((string)($r['type'] ?? '')), $type_filter_list, true);
                }));
              }

              $sorters = $this->parse_sort_spec(sanitize_text_field($atts['sort']));
              $this->sort_rows($rows, $sorters);
              ?>

              <?php foreach ($rows as $r): ?>
                <tr>
                  <td><?php echo $this->type_with_icon((string)$r['type']); ?></td>
                  <td><?php echo esc_html(number_format_i18n((int)$r['count'], 0)); ?></td>
                  <td><?php echo $this->dash_if_zero($r['km'], 2); ?></td>
                  <td><?php echo $this->dash_if_zero_int($r['hm']); ?></td>
                  <td>
                    <?php if (!empty($r['longest_id'])):
                      $url = 'https://www.strava.com/activities/' . rawurlencode((string)$r['longest_id']);
                    ?>
                      <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer">
                        <?php echo $this->dash_if_zero($r['longest_km'], 2); ?> km – <?php echo esc_html($r['longest_name']); ?>
                      </a>
                    <?php else: ?>
                      –
                    <?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        <?php endif; ?>
      </div>
    </div>
    <?php
    return ob_get_clean();
  }

  /**
   * Top activity card (by metric)
   */
  public function sc_top_activity($atts): string {
    wp_enqueue_style('sw-strava-frontend');

    $atts = shortcode_atts([
      'year' => 'current',
      'type' => 'Ride',
      'metric' => 'distance', // distance|elevation|moving_time
      'title' => '',
    ], $atts);

    // Title handling (supports %date, and title="none" to hide)
    $title_template = $this->normalize_title_template($atts['title'] ?? '');
    $hide_title = $this->is_title_none($title_template);

    $api = new SW_Strava_API();

    $year_spec = sanitize_text_field($atts['year']);
    $years = $api->resolve_years($year_spec);

    $type = sanitize_text_field($atts['type']);
    if ($type === '' || strtolower($type) === 'all') {
      return '<div class="sw-strava-error">Bitte <code>type</code> angeben (z.B. Ride, Run).</div>';
    }

    $metric = strtolower(sanitize_text_field($atts['metric']));
    if (!in_array($metric, ['distance', 'elevation', 'moving_time'], true)) $metric = 'distance';

    $resp = $api->cached_activities_for_years($years);
    if (!$resp['ok']) return '<div class="sw-strava-error">' . esc_html($resp['error']) . '</div>';

    $acts = array_values(array_filter($resp['data'], function ($a) use ($type) {
      return isset($a['type']) && strtolower((string)$a['type']) === strtolower((string)$type);
    }));

    $dateText = $this->format_stats_year_title($year_spec, $years);

    if (empty($acts)) {
      $fallbackTitle = 'Top ' . $type . ' (' . $dateText . ')';
      $title = $hide_title ? '' : $this->build_title($title_template, $fallbackTitle, $dateText);
      $titleHtml = $hide_title ? '' : ('<div class="sw-title">' . esc_html($title) . '</div>');
      return '<div class="sw-strava sw-top-activity"' . $this->design_style_attr() . '><div class="sw-card">' . $titleHtml . '<div>Keine Aktivitäten gefunden.</div></div></div>';
    }

    $best = null;
    $bestVal = -INF;

    foreach ($acts as $a) {
      if ($metric === 'distance') {
        $val = floatval($a['distance'] ?? 0);
      } elseif ($metric === 'elevation') {
        $val = floatval($a['total_elevation_gain'] ?? 0);
      } else {
        $val = floatval($a['moving_time'] ?? 0);
      }

      if ($val > $bestVal) {
        $bestVal = $val;
        $best = $a;
      }
    }

    if (!$best) return '<div class="sw-strava-error">Konnte keine Top-Aktivität bestimmen.</div>';

    $name = $best['name'] ?? '';
    $atype = $best['type'] ?? $type;

    $km = floatval($best['distance'] ?? 0) / 1000;
    $hm = floatval($best['total_elevation_gain'] ?? 0);

    $moving_s = intval($best['moving_time'] ?? 0);
    $elapsed_s = intval($best['elapsed_time'] ?? 0);

    $dt = $best['start_date_local'] ?? ($best['start_date'] ?? '');
    $dt_fmt = $dt ? date_i18n('d.m.Y H:i', strtotime($dt)) : '–';

    $id = $best['id'] ?? '';
    $url = $id ? ('https://www.strava.com/activities/' . rawurlencode((string)$id)) : '';

    $metricLabel = ($metric === 'distance') ? 'Top Distanz' : (($metric === 'elevation') ? 'Top Höhenmeter' : 'Top Zeit');

    $fallbackTitle = $metricLabel . ' – ' . $dateText;
    $title = $hide_title ? '' : $this->build_title($title_template, $fallbackTitle, $dateText);

    ob_start(); ?>
    <div class="sw-strava sw-top-activity"<?php echo $this->design_style_attr(); ?>>
      <div class="sw-card">
        <?php if (!$hide_title): ?>
          <div class="sw-title"><?php echo esc_html($title); ?></div>
        <?php endif; ?>

        <div class="sw-top-main">
          <div class="sw-top-name">
            <?php if ($url): ?>
              <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html($name); ?></a>
            <?php else: ?>
              <?php echo esc_html($name); ?>
            <?php endif; ?>
          </div>

          <div class="sw-top-meta">
            <span class="sw-pill"><?php echo $this->type_with_icon($atype); ?></span>
            <span class="sw-pill"><?php echo esc_html($dt_fmt); ?></span>
          </div>
        </div>

        <div class="sw-top-kpis">
          <div class="sw-kpi"><div class="sw-kpi-val"><?php echo $this->dash_if_zero($km, 2); ?></div><div class="sw-kpi-lbl">km</div></div>
          <div class="sw-kpi"><div class="sw-kpi-val"><?php echo $this->dash_if_zero_int($hm); ?></div><div class="sw-kpi-lbl">hm</div></div>
          <div class="sw-kpi"><div class="sw-kpi-val"><?php echo esc_html($this->format_duration($moving_s)); ?></div><div class="sw-kpi-lbl">moving</div></div>
          <div class="sw-kpi"><div class="sw-kpi-val"><?php echo esc_html($this->format_duration($elapsed_s)); ?></div><div class="sw-kpi-lbl">elapsed</div></div>
        </div>
      </div>
    </div>
    <?php
    return ob_get_clean();
  }

  /**
   * Photos grid
   */
  public function sc_photos($atts): string {
    wp_enqueue_style('sw-strava-frontend');

    $atts = shortcode_atts([
      'count' => 10,
      'size' => 600,
    ], $atts);

    $count = intval($atts['count']);
    $size = intval($atts['size']);

    $api = new SW_Strava_API();
    $resp = $api->cached_recent_photos($count, $size);
    if (!$resp['ok']) return '<div class="sw-strava-error">' . esc_html($resp['error']) . '</div>';

    $photos = $resp['data'];
    if (empty($photos)) return '<div class="sw-strava">Keine Fotos gefunden.</div>';

    ob_start(); ?>
    <div class="sw-strava sw-strava-photos"<?php echo $this->design_style_attr(); ?>>
      <div class="sw-photos-grid">
        <?php foreach ($photos as $p):
          $aid = $p['activity_id'] ?? '';
          $act_url = $aid ? ('https://www.strava.com/activities/' . rawurlencode((string)$aid)) : '';
        ?>
          <a class="sw-photo" href="<?php echo esc_url($act_url); ?>" target="_blank" rel="noopener noreferrer" title="<?php echo esc_attr($p['activity_name'] ?? ''); ?>">
            <img src="<?php echo esc_url($p['thumb'] ?? $p['url']); ?>" loading="lazy" alt="">
          </a>
        <?php endforeach; ?>
      </div>
    </div>
    <?php
    return ob_get_clean();
  }

  /**
   * AJAX: fetch one page of activity rows (used by pagination + header sorting)
   */
  public function ajax_activities_page(): void {
    check_ajax_referer('sw_strava_ajax', 'nonce');

    $fields_raw = sanitize_text_field($_POST['fields'] ?? 'name,type,distance,total_elevation_gain,start_date');
    $fields = array_values(array_filter(array_map('trim', explode(',', $fields_raw))));
    if (empty($fields)) $fields = ['name','type','distance','total_elevation_gain','start_date'];

    $year_spec = sanitize_text_field($_POST['year'] ?? 'current');
    $type_spec = sanitize_text_field($_POST['type'] ?? 'all');
    $types = $this->parse_types_csv($type_spec);
    $per_page = max(1, min(100, intval($_POST['per_page'] ?? 20)));
    $page = max(1, intval($_POST['page'] ?? 1));

    $sort_field = sanitize_text_field($_POST['sort_field'] ?? 'start_date');
    $sort_dir = strtolower(sanitize_text_field($_POST['sort_dir'] ?? 'desc'));
    if ($sort_dir !== 'asc' && $sort_dir !== 'desc') $sort_dir = 'desc';

    $api = new SW_Strava_API();
    $years = $api->resolve_years($year_spec);

    $resp = $api->cached_activities_for_years($years);
    if (!$resp['ok']) {
      wp_send_json_error(['message' => $resp['error']]);
    }

    $acts = $resp['data'];

    if ($types !== ['all']) {
      $acts = array_values(array_filter($acts, function ($a) use ($types) {
        $t = strtolower((string)($a['type'] ?? ''));
        return $t !== '' && in_array($t, $types, true);
      }));
    }

    usort($acts, function($a, $b) use ($sort_field, $sort_dir) {
      $dir = ($sort_dir === 'desc') ? -1 : 1;

      if ($sort_field === 'start_date' || $sort_field === 'start_date_local') {
        $ta = strtotime($a['start_date'] ?? ($a['start_date_local'] ?? '')) ?: 0;
        $tb = strtotime($b['start_date'] ?? ($b['start_date_local'] ?? '')) ?: 0;
        return ($ta <=> $tb) * $dir;
      }

      $va = $a[$sort_field] ?? null;
      $vb = $b[$sort_field] ?? null;

      if (in_array($sort_field, ['distance','total_elevation_gain','moving_time','elapsed_time'], true)) {
        $fa = floatval($va ?? 0);
        $fb = floatval($vb ?? 0);
        return ($fa <=> $fb) * $dir;
      }

      $sa = (string)($va ?? '');
      $sb = (string)($vb ?? '');
      return strnatcasecmp($sa, $sb) * $dir;
    });

    $total = count($acts);
    $total_pages = max(1, (int)ceil($total / $per_page));
    if ($page > $total_pages) $page = $total_pages;

    $offset = ($page - 1) * $per_page;
    $slice = array_slice($acts, $offset, $per_page);

    ob_start();
    foreach ($slice as $a) {
      $name = $a['name'] ?? '';
      $atype = $a['type'] ?? '';
      $km = floatval($a['distance'] ?? 0) / 1000;
      $elev = floatval($a['total_elevation_gain'] ?? 0);
      $dt = $a['start_date_local'] ?? ($a['start_date'] ?? '');
      $dt_fmt = $dt ? date_i18n('d.m.Y H:i', strtotime($dt)) : '–';

      $id = $a['id'] ?? '';
      $url = $id ? ('https://www.strava.com/activities/' . rawurlencode((string)$id)) : '';
      ?>
      <tr>
        <td>
          <?php if ($url): ?>
            <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html($name); ?></a>
          <?php else: ?>
            <?php echo esc_html($name); ?>
          <?php endif; ?>
        </td>
        <td><?php echo $this->type_with_icon($atype); ?></td>
        <td><?php echo $this->dash_if_zero($km, 2); ?></td>
        <td><?php echo $this->dash_if_zero_int($elev); ?></td>
        <td><?php echo esc_html($dt_fmt); ?></td>
      </tr>
      <?php
    }
    $rows_html = ob_get_clean();

    wp_send_json_success([
      'rows_html' => $rows_html,
      'page' => $page,
      'total_pages' => $total_pages,
      'total' => $total,
    ]);
  }

  // ----------------------------
  // Helpers
  // ----------------------------

  /**
   * Returns an inline style attribute that defines CSS variables from admin settings.
   * This makes the design settings work even if a caching/minify setup strips inline styles added during enqueue.
   */
  private function design_style_attr(): string {
    $opt = (new SW_Strava_API())->get_options();

    $vars = [
      '--sw-primary' => $opt['color_primary'] ?? '',
      '--sw-table-head-bg' => $opt['color_table_head_bg'] ?? '',
      '--sw-table-head-fg' => $opt['color_table_head_fg'] ?? '',
      '--sw-row-alt' => $opt['color_row_alt'] ?? '',
      '--sw-widget-bg' => $opt['color_widget_bg'] ?? '',
      '--sw-title-bg' => $opt['color_title_bg'] ?? '',
      '--sw-title-fg' => $opt['color_title_fg'] ?? '',
      '--sw-border' => $opt['color_border'] ?? '',
    ];

    $style_parts = [];
    foreach ($vars as $k => $v) {
      $v = trim((string)$v);
      if ($v === '') continue;
      // Basic allow-list: hex colors or rgb/rgba/hsl/hsla or 'transparent'
      if (!preg_match('/^#([0-9a-fA-F]{3}|[0-9a-fA-F]{6}|[0-9a-fA-F]{8})$/', $v)
          && !preg_match('/^(rgb|rgba|hsl|hsla)\([^\)]+\)$/', $v)
          && strtolower($v) !== 'transparent') {
        continue;
      }
      $style_parts[] = $k . ':' . $v;
    }

    if (empty($style_parts)) return '';
    return ' style="' . esc_attr(implode(';', $style_parts)) . '"';
  }

  // Backwards compatible alias (some parts of your code called this)
  private function type_icon_html(string $type): string {
    return $this->type_with_icon($type);
  }

  private function type_with_icon(string $type): string {
    $t = strtolower(trim($type));

    // 1) Admin-defined icon map (attachment IDs) in options
    $opt = (new SW_Strava_API())->get_options();
    $icon_map = isset($opt['icon_map']) && is_array($opt['icon_map']) ? $opt['icon_map'] : [];
    $att_id = $icon_map[$t] ?? ($icon_map[$type] ?? null);
    $att_id = $att_id ? intval($att_id) : 0;

    if ($att_id > 0) {
      $url = wp_get_attachment_url($att_id);
      if ($url) {
        return '<span class="sw-type">'
          . '<img class="sw-type-icon" src="' . esc_url($url) . '" alt="" loading="lazy" /> '
          . '<span class="sw-type-text">' . esc_html($type) . '</span>'
          . '</span>';
      }
    }

    // 2) Fallback to Dashicons
    $map = [
      'run' => 'dashicons-controls-play',
      'trailrun' => 'dashicons-controls-play',
      'ride' => 'dashicons-bike',
      'virtualride' => 'dashicons-bike',
      'ebikeride' => 'dashicons-bike',
      'walk' => 'dashicons-location-alt',
      'hike' => 'dashicons-location-alt',
      'swim' => 'dashicons-admin-site-alt3',
      'rowing' => 'dashicons-image-flip-horizontal',
      'kayaking' => 'dashicons-image-flip-horizontal',
      'workout' => 'dashicons-heart',
      'weighttraining' => 'dashicons-universal-access-alt',
      'yoga' => 'dashicons-smiley',
    ];

    $icon = $map[$t] ?? 'dashicons-marker';

    return '<span class="sw-type">'
      . '<span class="dashicons ' . esc_attr($icon) . '"></span> '
      . '<span class="sw-type-text">' . esc_html($type) . '</span>'
      . '</span>';
  }

  private function format_stats_year_title(string $year_spec, array $years): string {
    if (count($years) === 1) return (string)$years[0];
    return min($years) . ' bis ' . max($years);
  }

  private function dash_if_zero($val, int $decimals = 2): string {
    if ($val === null || $val === '') return '–';
    $f = floatval($val);
    if (abs($f) < 0.0000001) return '–';
    return esc_html(number_format_i18n($f, $decimals));
  }

  private function dash_if_zero_int($val): string {
    if ($val === null || $val === '') return '–';
    $i = (int)round(floatval($val));
    if ($i === 0) return '–';
    return esc_html(number_format_i18n($i, 0));
  }

  private function format_duration($seconds): string {
    $s = intval($seconds);
    if ($s <= 0) return '–';
    $h = intdiv($s, 3600);
    $m = intdiv($s % 3600, 60);
    $sec = $s % 60;
    return ($h > 0)
      ? sprintf('%d:%02d:%02d', $h, $m, $sec)
      : sprintf('%d:%02d', $m, $sec);
  }

  private function parse_sort_spec(string $spec): array {
    $spec = trim($spec);
    if ($spec === '') return [['field' => 'count', 'dir' => 'desc'], ['field' => 'km', 'dir' => 'desc']];

    $spec = str_replace(';', ',', $spec);
    $parts = array_map('trim', explode(',', $spec));

    $out = [];
    foreach ($parts as $p) {
      if ($p === '') continue;

      if ($p[0] === '-') {
        $out[] = ['field' => strtolower(trim(substr($p, 1))), 'dir' => 'desc'];
        continue;
      }

      $tokens = preg_split('/\s+/', $p);
      $field = strtolower($tokens[0] ?? '');
      $dir = strtolower($tokens[1] ?? 'asc');
      if ($dir !== 'asc' && $dir !== 'desc') $dir = 'asc';
      $out[] = ['field' => $field, 'dir' => $dir];
    }

    return !empty($out) ? $out : [['field' => 'count', 'dir' => 'desc'], ['field' => 'km', 'dir' => 'desc']];
  }

  private function sort_rows(array &$rows, array $sorters): void {
    usort($rows, function($a, $b) use ($sorters) {
      foreach ($sorters as $s) {
        $f = $s['field'];
        $dir = $s['dir'];

        $va = $a[$f] ?? null;
        $vb = $b[$f] ?? null;

        if ($f === 'type') {
          $cmp = strnatcasecmp((string)$va, (string)$vb);
        } else {
          $cmp = (float)$va <=> (float)$vb;
        }

        if ($cmp !== 0) return $dir === 'desc' ? -$cmp : $cmp;
      }
      return 0;
    });
  }

  private function build_title(string $template, string $fallback, string $dateText): string {
    $template = trim($template);
    if ($template === '') return $fallback;

    // %date Platzhalter ersetzen
    $out = str_replace('%date', $dateText, $template);

    // Optional: doppelte Spaces glätten
    $out = preg_replace('/\s+/', ' ', $out);

    return trim($out);
  }

}
