<?php
if (!defined('ABSPATH')) exit;

class SW_Strava_API {

  private $api_base = 'https://www.strava.com/api/v3';
  private $oauth_token_url = 'https://www.strava.com/oauth/token';

  public function get_options(): array {
    $opt = get_option(SW_STRAVA_OPT, []);
    return is_array($opt) ? $opt : [];
  }

  public function update_options(array $new): void {
    update_option(SW_STRAVA_OPT, $new, false);
  }

  private function get_access_token(): ?string {
    $opt = $this->get_options();
    if (empty($opt['access_token'])) return null;

    // Refresh if expiring within 5 minutes
    $expires_at = isset($opt['expires_at']) ? intval($opt['expires_at']) : 0;
    if ($expires_at && $expires_at < (time() + 300)) {
      $this->refresh_token();
      $opt = $this->get_options();
    }

    return $opt['access_token'] ?? null;
  }

  public function exchange_code_for_token(string $code): array {
    $opt = $this->get_options();
    $client_id = $opt['client_id'] ?? '';
    $client_secret = $opt['client_secret'] ?? '';
    $redirect_uri = $this->get_redirect_uri();

    $resp = wp_remote_post($this->oauth_token_url, [
      'timeout' => 20,
      'body' => [
        'client_id' => $client_id,
        'client_secret' => $client_secret,
        'code' => $code,
        'grant_type' => 'authorization_code',
      ],
    ]);

    if (is_wp_error($resp)) return ['ok' => false, 'error' => $resp->get_error_message()];

    $data = json_decode(wp_remote_retrieve_body($resp), true);
    if (!is_array($data) || empty($data['access_token'])) {
      return ['ok' => false, 'error' => 'Token exchange failed. Response: ' . wp_remote_retrieve_body($resp)];
    }

    // Persist latest refresh_token (Strava can rotate it)
    $opt['access_token']  = $data['access_token'];
    $opt['refresh_token'] = $data['refresh_token'] ?? ($opt['refresh_token'] ?? '');
    $opt['expires_at']    = $data['expires_at'] ?? 0;

    if (!empty($data['athlete']['id'])) {
      $opt['athlete_id'] = $data['athlete']['id'];
    }

    $this->update_options($opt);
    return ['ok' => true, 'data' => $data];
  }

  public function refresh_token(): array {
    $opt = $this->get_options();
    if (empty($opt['client_id']) || empty($opt['client_secret']) || empty($opt['refresh_token'])) {
      return ['ok' => false, 'error' => 'Missing client_id/client_secret/refresh_token'];
    }

    $resp = wp_remote_post($this->oauth_token_url, [
      'timeout' => 20,
      'body' => [
        'client_id' => $opt['client_id'],
        'client_secret' => $opt['client_secret'],
        'grant_type' => 'refresh_token',
        'refresh_token' => $opt['refresh_token'],
      ],
    ]);

    if (is_wp_error($resp)) return ['ok' => false, 'error' => $resp->get_error_message()];

    $data = json_decode(wp_remote_retrieve_body($resp), true);
    if (!is_array($data) || empty($data['access_token'])) {
      return ['ok' => false, 'error' => 'Refresh failed. Response: ' . wp_remote_retrieve_body($resp)];
    }

    $opt['access_token']  = $data['access_token'];
    $opt['refresh_token'] = $data['refresh_token'] ?? $opt['refresh_token'];
    $opt['expires_at']    = $data['expires_at'] ?? $opt['expires_at'];

    $this->update_options($opt);
    return ['ok' => true, 'data' => $data];
  }

  public function get_redirect_uri(): string {
    return admin_url('admin-post.php?action=sw_strava_oauth_callback');
  }

  public function get_authorize_url(): string {
    $opt = $this->get_options();
    $client_id = $opt['client_id'] ?? '';
    $redirect_uri = $this->get_redirect_uri();

    // Scopes: activity:read for public/followers; activity:read_all if you want "Only Me" too
    $scope = !empty($opt['scope']) ? $opt['scope'] : 'read,activity:read_all';

    $state = wp_create_nonce('sw_strava_oauth_state');

    return add_query_arg([
      'client_id' => $client_id,
      'response_type' => 'code',
      'redirect_uri' => $redirect_uri,
      'approval_prompt' => 'auto',
      'scope' => $scope,
      'state' => $state,
    ], 'https://www.strava.com/oauth/authorize');
  }

  private function api_get(string $path, array $query = []): array {
    $token = $this->get_access_token();
    if (!$token) return ['ok' => false, 'error' => 'Not connected to Strava'];

    $url = add_query_arg($query, $this->api_base . $path);

    $resp = wp_remote_get($url, [
      'timeout' => 20,
      'headers' => [
        'Authorization' => 'Bearer ' . $token,
      ],
    ]);

    if (is_wp_error($resp)) return ['ok' => false, 'error' => $resp->get_error_message()];

    $code = wp_remote_retrieve_response_code($resp);
    $body = wp_remote_retrieve_body($resp);
    $data = json_decode($body, true);

    if ($code >= 400) return ['ok' => false, 'error' => "API error ($code): $body"];
    return ['ok' => true, 'data' => $data];
  }

  public function list_activities(int $after = 0, int $before = 0, int $page = 1, int $per_page = 30): array {
    $q = ['page' => $page, 'per_page' => $per_page];
    if ($after > 0) $q['after'] = $after;
    if ($before > 0) $q['before'] = $before;

    return $this->api_get('/athlete/activities', $q);
  }

  public function athlete_stats(): array {
    $opt = $this->get_options();
    $id = isset($opt['athlete_id']) ? intval($opt['athlete_id']) : 0;
    if (!$id) return ['ok' => false, 'error' => 'Missing athlete_id (connect again).'];

    return $this->api_get('/athletes/' . $id . '/stats');
  }

  public function cached_activities_for_year(int $year): array {
    $key = 'sw_strava_activities_' . $year;
    $cached = get_transient($key);
    if (is_array($cached)) return ['ok' => true, 'data' => $cached, 'cached' => true];

    // Fetch all pages for the year
    $after = strtotime($year . '-01-01 00:00:00');
    $before = strtotime(($year + 1) . '-01-01 00:00:00');

    $all = [];
    $page = 1;

    while (true) {
      $resp = $this->list_activities($after, $before, $page, 50);
      if (!$resp['ok']) return $resp;

      $items = $resp['data'];
      if (!is_array($items) || count($items) === 0) break;

      $all = array_merge($all, $items);
      if (count($items) < 50) break;
      $page++;
      if ($page > 50) break; // safety
    }

    set_transient($key, $all, 60 * 60); // 1h cache
    return ['ok' => true, 'data' => $all, 'cached' => false];
  }

public function compute_stats_from_activities(array $acts, int $year): array {
  $total_m = 0.0;
  $total_elev = 0.0;
  $by_type = [];
  $longest_by_type = [];

  foreach ($acts as $a) {
    $type = $a['type'] ?? 'Unknown';
    $dist = floatval($a['distance'] ?? 0); // meters
    $elev = floatval($a['total_elevation_gain'] ?? 0);

    $total_m += $dist;
    $total_elev += $elev;

    if (!isset($by_type[$type])) {
      $by_type[$type] = ['count' => 0, 'distance_m' => 0.0, 'elev_m' => 0.0];
    }

    $by_type[$type]['count']++;
    $by_type[$type]['distance_m'] += $dist;
    $by_type[$type]['elev_m'] += $elev;

    if (!isset($longest_by_type[$type]) || $dist > ($longest_by_type[$type]['distance_m'] ?? 0)) {
      $longest_by_type[$type] = [
        'name' => $a['name'] ?? '',
        'id' => $a['id'] ?? '',
        'distance_m' => $dist,
        'start_date' => $a['start_date_local'] ?? ($a['start_date'] ?? ''),
      ];
    }
  }

  return [
    'year' => $year,
    'total_km' => round($total_m / 1000, 2),
    'total_elev_m' => round($total_elev, 0),
    'by_type' => $by_type,
    'longest_by_type' => $longest_by_type,
  ];
}

  public function resolve_years(string $year_spec): array {
  $year_spec = trim(strtolower($year_spec));
  $current = intval(gmdate('Y'));
  $previous = $current - 1;

  if ($year_spec === 'current') {
    return [$current];
  }

  if ($year_spec === 'previous') {
    return [$previous];
  }

  // Range: "2023..current" oder "2023..2025"
  if (strpos($year_spec, '..') !== false) {
    [$a, $b] = array_map('trim', explode('..', $year_spec, 2));
  $start = ($a === 'current') ? $current : (($a === 'previous') ? $previous : intval($a));
  $end   = ($b === 'current') ? $current : (($b === 'previous') ? $previous : intval($b));

    if ($start <= 0 || $end <= 0) return [$current];

    if ($start > $end) { $tmp = $start; $start = $end; $end = $tmp; }

    $years = [];
    for ($y = $start; $y <= $end; $y++) $years[] = $y;
    return $years;
  }

  // Single year
  $y = intval($year_spec);
  return $y > 0 ? [$y] : [$current];
}

public function cached_activities_for_years(array $years): array {
  $all = [];
  foreach ($years as $y) {
    $r = $this->cached_activities_for_year(intval($y));
    if (!$r['ok']) return $r;
    if (!empty($r['data']) && is_array($r['data'])) {
      $all = array_merge($all, $r['data']);
    }
  }

  // sort newest-first by start_date (fallback: start_date_local)
  usort($all, function($a, $b) {
    $da = strtotime($a['start_date'] ?? ($a['start_date_local'] ?? '')) ?: 0;
    $db = strtotime($b['start_date'] ?? ($b['start_date_local'] ?? '')) ?: 0;
    return $db <=> $da;
  });

  return ['ok' => true, 'data' => $all, 'cached' => true];
}

public function list_activity_photos(int $activity_id, int $size = 600): array {
  return $this->api_get('/activities/' . intval($activity_id) . '/photos', [
    'size' => max(100, min(5000, $size)),
    'photo_sources' => 'true',
  ]);
}


public function list_recent_activities(int $per_page = 30): array {
  return $this->api_get('/athlete/activities', [
    'page' => 1,
    'per_page' => max(1, min(100, $per_page)),
  ]);
}

public function cached_recent_photos(int $count, int $size): array {
  $count = max(1, min(30, $count));
  $size  = max(100, min(2048, $size));

  $key = 'sw_strava_recent_photos_' . $count . '_' . $size;
  $cached = get_transient($key);
  if (is_array($cached)) return ['ok' => true, 'data' => $cached, 'cached' => true];

  // hole mehr Activities als Photos, weil nicht jede Aktivität ein Foto hat
  $acts_resp = $this->list_recent_activities(50);
  if (!$acts_resp['ok']) return $acts_resp;

  $photos = [];
  foreach ($acts_resp['data'] as $a) {
    $aid = intval($a['id'] ?? 0);
    if (!$aid) continue;

    $pr = $this->list_activity_photos($aid, $size);
    if (!$pr['ok']) continue;

    if (is_array($pr['data'])) {
      foreach ($pr['data'] as $p) {
        // Felder können variieren, häufig gibt es urls/thumbnail/unique_id.
$url = $this->pick_best_photo_url($p, $size);

// Thumb: möglichst klein, falls vorhanden, sonst best match
$thumb = null;
if (!empty($p['urls']) && is_array($p['urls'])) {
  $thumb = $p['urls']['100'] ?? $p['urls']['200'] ?? null;
}
if (!$thumb) $thumb = $url;

        if ($url) {
          $photos[] = [
            'url' => $url,
            'thumb' => $thumb ?: $url,
            'activity_id' => $aid,
            'activity_name' => $a['name'] ?? '',
          ];
        }
        if (count($photos) >= $count) break 2;
      }
    }
  }

  set_transient($key, $photos, 60 * 30); // 30 min
  return ['ok' => true, 'data' => $photos, 'cached' => false];
}

private function pick_best_photo_url(array $photo, int $requested): ?string {
  if (empty($photo['urls']) || !is_array($photo['urls'])) return null;

  $urls = $photo['urls'];

  // Keys sind oft Strings wie "5000", "2048", ...
  $bestKey = null;
  $bestDiff = PHP_INT_MAX;

  foreach ($urls as $k => $u) {
    $ki = intval($k);
    if (!$u) continue;

    $diff = abs($ki - $requested);
    if ($diff < $bestDiff) {
      $bestDiff = $diff;
      $bestKey = $k;
    }
  }

  if ($bestKey !== null && !empty($urls[$bestKey])) return $urls[$bestKey];

  // Fallback: erste URL
  foreach ($urls as $u) {
    if (!empty($u)) return $u;
  }

  return null;
}


}
