<?php
if (!defined('ABSPATH')) exit;

class SW_Strava_Admin {

  public function init(): void {
    add_action('admin_menu', [$this, 'menu']);
    add_action('admin_post_sw_strava_save_settings', [$this, 'save_settings']);
    add_action('admin_post_sw_strava_oauth_callback', [$this, 'oauth_callback']);
    add_action('sw_strava_cron_refresh_cache', [$this, 'cron_refresh_cache']);

    // Icon map uploader
    add_action('admin_enqueue_scripts', [$this, 'admin_assets']);
    add_filter('upload_mimes', [$this, 'allow_svg_mime']);
  }

  public function menu(): void {
    add_options_page('Strava Widgets', 'Strava Widgets', 'manage_options', 'sw-strava', [$this, 'page']);
  }

  public function admin_assets($hook): void {
    // Nur auf unserer Settings-Seite
    if ($hook !== 'settings_page_sw-strava') return;
    wp_enqueue_media();
  }

  /**
   * Optional: SVG Upload erlauben (nur Admins). SVG kann sicherheitssensibel sein.
   */
  public function allow_svg_mime($mimes) {
    if (current_user_can('manage_options')) {
      $mimes['svg'] = 'image/svg+xml';
    }
    return $mimes;
  }

  public function page(): void {
    if (!current_user_can('manage_options')) return;

    $api = new SW_Strava_API();
    $opt = $api->get_options();

    // Design defaults
    $opt['color_primary'] = $opt['color_primary'] ?? '#2f6fed';
    $opt['color_table_head_bg'] = $opt['color_table_head_bg'] ?? $opt['color_primary'];
    $opt['color_table_head_fg'] = $opt['color_table_head_fg'] ?? '#ffffff';
    $opt['color_row_alt'] = $opt['color_row_alt'] ?? '#eef4ff';
    $opt['color_widget_bg'] = $opt['color_widget_bg'] ?? '#ffffff';
    $opt['color_title_bg'] = $opt['color_title_bg'] ?? $opt['color_primary'];
    $opt['color_title_fg'] = $opt['color_title_fg'] ?? '#ffffff';
    $opt['color_border'] = $opt['color_border'] ?? '#e5e5e5';

    $auth_url = '';
    if (!empty($opt['client_id'])) {
      $auth_url = $api->get_authorize_url();
    }

    ?>
    <div class="wrap">
      <h1>Strava Widgets</h1>
      <p>Verbinde Strava via OAuth2, um Aktivitäten & Statistiken im Frontend anzuzeigen.</p>

      <h2>API Settings</h2>
      <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <input type="hidden" name="action" value="sw_strava_save_settings" />
        <?php wp_nonce_field('sw_strava_save_settings'); ?>

        <table class="form-table" role="presentation">
          <tr>
            <th scope="row"><label for="client_id">Client ID</label></th>
            <td><input name="client_id" id="client_id" class="regular-text" value="<?php echo esc_attr($opt['client_id'] ?? ''); ?>" /></td>
          </tr>
          <tr>
            <th scope="row"><label for="client_secret">Client Secret</label></th>
            <td><input name="client_secret" id="client_secret" class="regular-text" value="<?php echo esc_attr($opt['client_secret'] ?? ''); ?>" /></td>
          </tr>
          <tr>
            <th scope="row"><label for="scope">Scope</label></th>
            <td>
              <input name="scope" id="scope" class="regular-text" value="<?php echo esc_attr($opt['scope'] ?? 'read,activity:read_all'); ?>" />
              <p class="description">Empfohlen: <code>read,activity:read_all</code> (damit auch „Only Me“ Aktivitäten möglich sind).</p>
            </td>
          </tr>
          <tr>
            <th scope="row">Callback URL</th>
            <td><code><?php echo esc_html($api->get_redirect_uri()); ?></code></td>
          </tr>

          <tr>
            <th scope="row">Design</th>
            <td>
              <p class="description">Diese Farben steuern Tabellen (Header + Zebra), Widget-Box-Hintergrund und Titel-Hintergrund.</p>
              <table style="border-collapse:collapse;">
                <tr>
                  <td style="padding:4px 16px 4px 0;"><label for="color_primary">Primary</label></td>
                  <td><input type="color" id="color_primary" name="color_primary" value="<?php echo esc_attr($opt['color_primary']); ?>"></td>
                </tr>
                <tr>
                  <td style="padding:4px 16px 4px 0;"><label for="color_table_head_bg">Tabellenkopf Hintergrund</label></td>
                  <td><input type="color" id="color_table_head_bg" name="color_table_head_bg" value="<?php echo esc_attr($opt['color_table_head_bg']); ?>"></td>
                </tr>
                <tr>
                  <td style="padding:4px 16px 4px 0;"><label for="color_table_head_fg">Tabellenkopf Text</label></td>
                  <td><input type="color" id="color_table_head_fg" name="color_table_head_fg" value="<?php echo esc_attr($opt['color_table_head_fg']); ?>"></td>
                </tr>
                <tr>
                  <td style="padding:4px 16px 4px 0;"><label for="color_row_alt">Zebra-Zeile (jede 2.)</label></td>
                  <td><input type="color" id="color_row_alt" name="color_row_alt" value="<?php echo esc_attr($opt['color_row_alt']); ?>"></td>
                </tr>
                <tr>
                  <td style="padding:4px 16px 4px 0;"><label for="color_widget_bg">Widget Hintergrund</label></td>
                  <td><input type="color" id="color_widget_bg" name="color_widget_bg" value="<?php echo esc_attr($opt['color_widget_bg']); ?>"></td>
                </tr>
                <tr>
                  <td style="padding:4px 16px 4px 0;"><label for="color_title_bg">Title Hintergrund (sw-title)</label></td>
                  <td><input type="color" id="color_title_bg" name="color_title_bg" value="<?php echo esc_attr($opt['color_title_bg']); ?>"></td>
                </tr>
                <tr>
                  <td style="padding:4px 16px 4px 0;"><label for="color_title_fg">Title Textfarbe</label></td>
                  <td><input type="color" id="color_title_fg" name="color_title_fg" value="<?php echo esc_attr($opt['color_title_fg']); ?>"></td>
                </tr>
                <tr>
                  <td style="padding:4px 16px 4px 0;"><label for="color_border">Border</label></td>
                  <td><input type="color" id="color_border" name="color_border" value="<?php echo esc_attr($opt['color_border']); ?>"></td>
                </tr>
              </table>
            </td>
          </tr>

          <tr>
            <th scope="row">Icon Map</th>
            <td>
              <p class="description">
                Ordne je Aktivitätstyp (z.B. <code>ride</code>, <code>run</code>) ein Icon aus der Mediathek zu (SVG/PNG/WebP).
                Das Icon wird im Frontend als Prefix vor dem Typ angezeigt.
              </p>

              <?php
              $icon_map = isset($opt['icon_map']) && is_array($opt['icon_map']) ? $opt['icon_map'] : [];
              $defaults = ['ride', 'run', 'swim', 'walk', 'workout'];
              foreach ($defaults as $k) {
                if (!isset($icon_map[$k])) $icon_map[$k] = 0;
              }
              ?>

              <table class="widefat striped" style="max-width: 900px;">
                <thead>
                  <tr>
                    <th style="width:220px;">Type (Key)</th>
                    <th>Icon</th>
                    <th style="width:120px;">Aktion</th>
                  </tr>
                </thead>

                <tbody id="sw-strava-iconmap">
                  <?php foreach ($icon_map as $type_key => $att_id):
                    $type_key = (string)$type_key;
                    $att_id = intval($att_id);
                    $url = $att_id ? wp_get_attachment_url($att_id) : '';
                    ?>
                    <tr class="sw-icon-row">
                      <td>
                        <input class="regular-text" name="icon_type[]" value="<?php echo esc_attr($type_key); ?>" placeholder="ride" />
                        <p class="description">Empfohlen: klein geschrieben (z.B. <code>ride</code>)</p>
                      </td>

                      <td>
                        <input type="hidden" class="sw-icon-attachment" name="icon_attachment[]" value="<?php echo esc_attr($att_id); ?>" />
                        <button type="button" class="button sw-icon-upload">Aus Medien wählen</button>
                        <button type="button" class="button sw-icon-remove">Entfernen</button>

                        <div class="sw-icon-preview" style="margin-top:8px;">
                          <?php if ($url): ?>
                            <img src="<?php echo esc_url($url); ?>" style="width:20px;height:20px;object-fit:contain;vertical-align:middle;" alt="" />
                            <code style="margin-left:8px;"><?php echo esc_html($url); ?></code>
                          <?php else: ?>
                            <span style="opacity:.7;">(kein Icon gesetzt)</span>
                          <?php endif; ?>
                        </div>
                      </td>

                      <td>
                        <button type="button" class="button-link-delete sw-icon-row-delete">Zeile löschen</button>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>

              <p><button type="button" class="button" id="sw-icon-add">+ Type hinzufügen</button></p>

              <script>
                (function($){
                  function refreshPreview($row, url){
                    var $p = $row.find('.sw-icon-preview');
                    if(url){
                      $p.html('<img src="'+url+'" style="width:20px;height:20px;object-fit:contain;vertical-align:middle;" alt="" /> <code style="margin-left:8px;">'+url+'</code>');
                    } else {
                      $p.html('<span style="opacity:.7;">(kein Icon gesetzt)</span>');
                    }
                  }

                  $(document).on('click', '.sw-icon-upload', function(e){
                    e.preventDefault();
                    var $row = $(this).closest('tr');
                    var frame = wp.media({ title: 'Icon auswählen', button: { text: 'Übernehmen' }, multiple: false });
                    frame.on('select', function(){
                      var att = frame.state().get('selection').first().toJSON();
                      $row.find('.sw-icon-attachment').val(att.id);
                      refreshPreview($row, att.url);
                    });
                    frame.open();
                  });

                  $(document).on('click', '.sw-icon-remove', function(e){
                    e.preventDefault();
                    var $row = $(this).closest('tr');
                    $row.find('.sw-icon-attachment').val('');
                    refreshPreview($row, '');
                  });

                  $(document).on('click', '.sw-icon-row-delete', function(e){
                    e.preventDefault();
                    $(this).closest('tr').remove();
                  });

                  $('#sw-icon-add').on('click', function(){
                    var html =
                      '<tr class="sw-icon-row">' +
                        '<td>' +
                          '<input class="regular-text" name="icon_type[]" value="" placeholder="ride" />' +
                          '<p class="description">Empfohlen: klein geschrieben (z.B. <code>ride</code>)</p>' +
                        '</td>' +
                        '<td>' +
                          '<input type="hidden" class="sw-icon-attachment" name="icon_attachment[]" value="" />' +
                          '<button type="button" class="button sw-icon-upload">Aus Medien wählen</button> ' +
                          '<button type="button" class="button sw-icon-remove">Entfernen</button>' +
                          '<div class="sw-icon-preview" style="margin-top:8px;"><span style="opacity:.7;">(kein Icon gesetzt)</span></div>' +
                        '</td>' +
                        '<td><button type="button" class="button-link-delete sw-icon-row-delete">Zeile löschen</button></td>' +
                      '</tr>';
                    $('#sw-strava-iconmap').append(html);
                  });
                })(jQuery);
              </script>
            </td>
          </tr>
        </table>

        <?php submit_button('Speichern'); ?>
      </form>

      <h2>Connection</h2>
      <?php if (!empty($opt['access_token'])): ?>
        <p><strong>Status:</strong> verbunden ✅ (Athlete ID: <?php echo esc_html($opt['athlete_id'] ?? 'n/a'); ?>)</p>
      <?php else: ?>
        <p><strong>Status:</strong> nicht verbunden ❌</p>
      <?php endif; ?>

      <?php if ($auth_url): ?>
        <p><a class="button button-primary" href="<?php echo esc_url($auth_url); ?>">Mit Strava verbinden</a></p>
      <?php else: ?>
        <p class="description">Bitte zuerst Client ID/Secret speichern.</p>
      <?php endif; ?>

      <h2>Shortcodes</h2>

      <p>Du kannst die Widgets entweder über das Widget-Menü oder über Shortcodes in Seiten/Beiträge einbinden.</p>

      <h3>1) Aktivitätenliste</h3>
      <p><code>[strava_activities]</code> zeigt eine tabellarische Liste von Aktivitäten inkl. Link zur Aktivität (öffnet in neuem Tab). Mit <code>per_page</code> wird automatisch eine AJAX-Pagination angezeigt, sobald mehr Einträge vorhanden sind.</p>

      <ul style="list-style: disc; margin-left: 20px;">
        <li><code>year</code>: <code>2025</code> | <code>current</code> | <code>previous</code> | <code>2023..current</code> | <code>2022..2024</code></li>
        <li><code>type</code>: <code>all</code> (default) oder z.B. <code>Ride</code>, <code>Run</code>, <code>Walk</code>, <code>Swim</code> … (Strava Activity Types). Auch möglich: CSV wie <code>Ride,Run,Swim</code></li>
        <li><code>per_page</code>: Anzahl pro Seite (1–100), default 20</li>
        <li><code>page</code>: Startseite (default 1)</li>
        <li><code>fields</code>: name,type,distance,total_elevation_gain,start_date,moving_time,elapsed_time</li>
        <li><code>title</code> (optional): <code>Auswertung %date</code> | <code>Meine Statistik %date</code> | <code>Ueberschrift ohne Datum</code></li>        
      </ul>

      <p><strong>Beispiele:</strong></p>
      <ul>
        <li><code>[strava_activities year="current" type="all" per_page="20"]</code></li>
        <li><code>[strava_activities year="2023..current" type="Ride" per_page="30"]</code></li>
        <li><code>[strava_activities year="2025" type="Run" per_page="15" page="1"]</code></li>
        <li><code>[strava_activities year="current" type="Workout" per_page="20" fields="name,type,moving_time,start_date"]</code></li>
        <li><code>[strava_activities year="2023..current" type="Ride,Run,Swim" per_page="10"]</code></li>
      </ul>

      <hr />

      <h3>2) Statistiken</h3>
      <p><code>[strava_stats]</code> berechnet Statistiken auf Basis der Aktivitäten (inkl. privater Aktivitäten, wenn Scope <code>activity:read_all</code> genutzt wird). Bei Gruppierung nach Typ kann die Tabelle mehrstufig sortiert werden.</p>

      <ul style="list-style: disc; margin-left: 20px;">
        <li><code>year</code>: <code>2025</code> | <code>current</code> | <code>previous</code> | <code>2020..current</code> | <code>2022..2024</code></li>
        <li><code>group</code>: <code>total</code> oder <code>type</code></li>
        <li><code>type</code> (optional, nur bei <code>group="type"</code>): <code>all</code> oder CSV wie <code>Ride,Run,Swim</code></li>
        <li><code>sort</code> (nur bei <code>group="type"</code>): Multi-Sort, z.B. <code>count desc, km desc</code> oder <code>type asc, km desc</code> oder Kurzform <code>count,-km</code></li>
        <li><code>metrics</code>: count,km,hm,moving_time,elapsed_time</li>
        <li><code>title</code> (optional): <code>Auswertung %date</code> | <code>Meine Statistik %date</code> | <code>Ueberschrift ohne Datum</code></li>        
      </ul>

      <p><strong>Beispiele:</strong></p>
      <ul>
        <li><code>[strava_stats year="2025" group="total"]</code></li>
        <li><code>[strava_stats year="current" group="type" sort="count desc, km desc"]</code></li>
        <li><code>[strava_stats year="2020..current" group="type" sort="type asc, km desc"]</code></li>
        <li><code>[strava_stats year="2023..2025" group="type" sort="count,-km"]</code></li>
        <li><code>[strava_stats year="2025" group="type" type="Ride,Run,Swim" sort="count desc, km desc"]</code></li>
      </ul>

      <p><em>Hinweis:</em> In der „Längste“-Spalte sind die Aktivitäten klickbar (öffnet Strava in neuem Tab).</p>

      <hr />

      <h3>3) Letzte Aktivität</h3>
      <p><code>[strava_latest]</code> zeigt die zuletzt vorhandene Aktivität als hervorgehobene Kachel. Der Aktivitätsname ist klickbar und öffnet die Aktivität auf Strava in einem neuen Tab.</p>

      <ul style="list-style: disc; margin-left: 20px;">
        <li><code>year</code>: optional (default current year). Du kannst auch <code>2023..current</code> angeben.</li>
      </ul>

      <p><strong>Beispiele:</strong></p>
      <ul>
        <li><code>[strava_latest]</code></li>
        <li><code>[strava_latest year="current"]</code></li>
        <li><code>[strava_latest year="previous"]</code></li>
        <li><code>[strava_latest year="2023..current"]</code></li>
      </ul>

      <hr />

      <h3>4) Letzte Bilder (Fotos)</h3>
      <p><code>[strava_photos]</code> zeigt die zuletzt gefundenen Strava-Fotos als Grid. Da Strava kein globales „Recent Photos“-Endpoint bereitstellt, werden Fotos aus den zuletzt geladenen Aktivitäten gesammelt.</p>

      <ul style="list-style: disc; margin-left: 20px;">
        <li><code>count</code>: Anzahl Bilder (1–30), default 10</li>
        <li><code>size</code>: Bildgröße (100–2048), default 600</li>
      </ul>

      <p><strong>Beispiele:</strong></p>
      <ul>
        <li><code>[strava_photos count="5"]</code></li>
        <li><code>[strava_photos count="10" size="600"]</code></li>
        <li><code>[strava_photos count="12" size="1000"]</code></li>
      </ul>

      <hr />

      <h3>5) Top-Aktivität</h3>
      <p><code>[strava_top_activity]</code> zeigt die “Top”-Aktivität innerhalb eines Zeitraums und eines Typs als hervorgehobene Box (inkl. km, hm, moving/elapsed time). Der Aktivitätsname ist klickbar und öffnet Strava im neuen Tab.</p>

      <ul style="list-style: disc; margin-left: 20px;">
        <li><code>year</code>: <code>2025</code> | <code>current</code> | <code>previous</code> | <code>2020..current</code> | <code>2022..2024</code></li>
        <li><code>type</code>: z.B. <code>Ride</code>, <code>Run</code>, <code>Walk</code>, <code>Swim</code>, <code>Workout</code>, <code>WeightTraining</code> …</li>
        <li><code>metric</code> (optional): <code>distance</code> (default) | <code>elevation</code> | <code>moving_time</code></li>
        <li><code>title</code> (optional): <code>Auswertung %date</code> | <code>Meine Statistik %date</code> | <code>Ueberschrift ohne Datum</code></li>
      </ul>

      <p><strong>Beispiele:</strong></p>
      <ul>
        <li><code>[strava_top_activity year="current" type="Ride"]</code> (längste Distanz im aktuellen Jahr)</li>
        <li><code>[strava_top_activity year="2020..current" type="Ride"]</code> (längste Distanz über mehrere Jahre)</li>
        <li><code>[strava_top_activity year="2025" type="Run" metric="elevation"]</code> (meiste Höhenmeter im Jahr 2025)</li>
        <li><code>[strava_top_activity year="2025" type="Workout" metric="moving_time"]</code> (längste Trainingszeit im Jahr 2025)</li>
      </ul>

    </div>
    <?php
  }

  public function save_settings(): void {
    if (!current_user_can('manage_options')) wp_die('Forbidden');
    check_admin_referer('sw_strava_save_settings');

    $api = new SW_Strava_API();
    $opt = $api->get_options();

    $opt['client_id'] = sanitize_text_field($_POST['client_id'] ?? '');
    $opt['client_secret'] = sanitize_text_field($_POST['client_secret'] ?? '');
    $opt['scope'] = sanitize_text_field($_POST['scope'] ?? 'read,activity:read_all');

    // Icon map
    $types = $_POST['icon_type'] ?? [];
    $atts  = $_POST['icon_attachment'] ?? [];

    $icon_map = [];
    if (is_array($types) && is_array($atts)) {
      $n = min(count($types), count($atts));
      for ($i = 0; $i < $n; $i++) {
        $k = strtolower(trim(sanitize_text_field($types[$i] ?? '')));
        if ($k === '') continue;

        $att_id = intval($atts[$i] ?? 0);
        if ($att_id > 0) {
          $icon_map[$k] = $att_id;
        }
      }
    }
    $opt['icon_map'] = $icon_map;

    // Design options
    $opt['color_primary'] = sanitize_hex_color($_POST['color_primary'] ?? '#2f6fed') ?: '#2f6fed';
    $opt['color_table_head_bg'] = sanitize_hex_color($_POST['color_table_head_bg'] ?? $opt['color_primary']) ?: $opt['color_primary'];
    $opt['color_table_head_fg'] = sanitize_hex_color($_POST['color_table_head_fg'] ?? '#ffffff') ?: '#ffffff';
    $opt['color_row_alt'] = sanitize_hex_color($_POST['color_row_alt'] ?? '#eef4ff') ?: '#eef4ff';
    $opt['color_widget_bg'] = sanitize_hex_color($_POST['color_widget_bg'] ?? '#ffffff') ?: '#ffffff';
    $opt['color_title_bg'] = sanitize_hex_color($_POST['color_title_bg'] ?? $opt['color_primary']) ?: $opt['color_primary'];
    $opt['color_title_fg'] = sanitize_hex_color($_POST['color_title_fg'] ?? '#ffffff') ?: '#ffffff';
    $opt['color_border'] = sanitize_hex_color($_POST['color_border'] ?? '#e5e5e5') ?: '#e5e5e5';

    $api->update_options($opt);

    wp_redirect(add_query_arg(['page' => 'sw-strava', 'saved' => 1], admin_url('options-general.php')));
    exit;
  }

  public function oauth_callback(): void {
    if (!current_user_can('manage_options')) wp_die('Forbidden');

    $state = sanitize_text_field($_GET['state'] ?? '');
    if (!wp_verify_nonce($state, 'sw_strava_oauth_state')) {
      wp_die('Invalid state');
    }

    $code = sanitize_text_field($_GET['code'] ?? '');
    if (!$code) wp_die('Missing code');

    $api = new SW_Strava_API();
    $res = $api->exchange_code_for_token($code);

    $url = add_query_arg(['page' => 'sw-strava', 'connected' => $res['ok'] ? 1 : 0], admin_url('options-general.php'));
    wp_redirect($url);
    exit;
  }

  public function cron_refresh_cache(): void {
    $api = new SW_Strava_API();
    // Refresh token (if needed)
    $api->refresh_token();

    // Warm cache for current year
    $year = intval(gmdate('Y'));
    $api->cached_activities_for_year($year);
  }
}
