<?php
/**
 * Plugin Name: WP Merge Fixer by dasFLOSEN
 * Description: Fixes thumbnail IDs, gallery IDs (Classic + Gutenberg), and regenerates attachment metadata in batches (no WP-CLI required).
 * Version: 0.1.69
 */

if (!defined('ABSPATH')) exit;

class WPMergeFixer {
  const CAP = 'manage_options';
  const MENU_SLUG = 'wp-merge-fixer';

  public static function init() {
    add_action('admin_menu', [__CLASS__, 'menu']);
  }

  public static function menu() {
    add_management_page(
      'WP Merge Fixer',
      'WP Merge Fixer',
      self::CAP,
      self::MENU_SLUG,
      [__CLASS__, 'page']
    );
  }

  private static function nonce_ok($action) {
    return isset($_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], $action);
  }

  private static function int($key, $default) {
    if (!isset($_POST[$key])) return $default;
    return intval($_POST[$key]);
  }

  public static function page() {
    if (!current_user_can(self::CAP)) {
      wp_die('Insufficient permissions.');
    }

    global $wpdb;

    $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 50000;
    $cutoff = isset($_POST['cutoff']) ? intval($_POST['cutoff']) : 50000; // imported posts >= cutoff
    $batch  = isset($_POST['batch']) ? max(10, intval($_POST['batch'])) : 200;

    $msg = '';
    $stats = null;

    // DRY RUN STATS
    if (isset($_POST['dry_run']) && self::nonce_ok('wpmf')) {
      $stats = [];

      // Imported posts that have thumbnail_id < cutoff (need +offset)
      $stats['thumb_needs_fix'] = (int) $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*)
        FROM {$wpdb->postmeta} pm
        JOIN {$wpdb->posts} p ON p.ID = pm.post_id
        WHERE pm.meta_key = '_thumbnail_id'
          AND p.ID >= %d
          AND pm.meta_value REGEXP '^[0-9]+$'
          AND CAST(pm.meta_value AS UNSIGNED) < %d
      ", $cutoff, $cutoff));

      // Old posts that were accidentally shifted (thumbnail_id >= cutoff)
      $stats['thumb_old_shifted'] = (int) $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*)
        FROM {$wpdb->postmeta} pm
        JOIN {$wpdb->posts} p ON p.ID = pm.post_id
        WHERE pm.meta_key = '_thumbnail_id'
          AND p.ID < %d
          AND pm.meta_value REGEXP '^[0-9]+$'
          AND CAST(pm.meta_value AS UNSIGNED) >= %d
      ", $cutoff, $cutoff));

      // Posts with classic gallery shortcode
      $stats['classic_gallery_posts'] = (int) $wpdb->get_var("
        SELECT COUNT(*)
        FROM {$wpdb->posts}
        WHERE post_type IN ('post','page')
          AND post_status IN ('publish','draft','private','future')
          AND post_content LIKE '%[gallery%'
      ");

      // Posts with Gutenberg gallery ids pattern
      $stats['gutenberg_gallery_posts'] = (int) $wpdb->get_var("
        SELECT COUNT(*)
        FROM {$wpdb->posts}
        WHERE post_type IN ('post','page')
          AND post_status IN ('publish','draft','private','future')
          AND post_content REGEXP '\"ids\"[[:space:]]*:[[:space:]]*\\[[0-9]'
      ");

      // Attachments missing metadata
      $stats['attachments_missing_meta'] = (int) $wpdb->get_var("
        SELECT COUNT(*)
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} m
          ON m.post_id = p.ID AND m.meta_key = '_wp_attachment_metadata'
        WHERE p.post_type = 'attachment'
          AND m.post_id IS NULL
      ");

      $msg = 'Dry-run completed.';
    }

    // FIX THUMBNAILS FOR IMPORTED POSTS ONLY (add offset)
    if (isset($_POST['fix_thumb_imported']) && self::nonce_ok('wpmf')) {
      // Only imported posts >= cutoff, and only values that still look unshifted (< cutoff)
      $affected = $wpdb->query($wpdb->prepare("
        UPDATE {$wpdb->postmeta} pm
        JOIN {$wpdb->posts} p ON p.ID = pm.post_id
        SET pm.meta_value = CAST(pm.meta_value AS UNSIGNED) + %d
        WHERE pm.meta_key = '_thumbnail_id'
          AND p.ID >= %d
          AND pm.meta_value REGEXP '^[0-9]+$'
          AND CAST(pm.meta_value AS UNSIGNED) < %d
        LIMIT %d
      ", $offset, $cutoff, $cutoff, $batch));
      $msg = "Fixed imported featured images (batch). Rows affected: {$affected}. Run again if needed.";
    }

    // UNDO THUMBNAIL SHIFT FOR OLD POSTS (subtract offset)
    if (isset($_POST['undo_thumb_old']) && self::nonce_ok('wpmf')) {
      $affected = $wpdb->query($wpdb->prepare("
        UPDATE {$wpdb->postmeta} pm
        JOIN {$wpdb->posts} p ON p.ID = pm.post_id
        SET pm.meta_value = CAST(pm.meta_value AS UNSIGNED) - %d
        WHERE pm.meta_key = '_thumbnail_id'
          AND p.ID < %d
          AND pm.meta_value REGEXP '^[0-9]+$'
          AND CAST(pm.meta_value AS UNSIGNED) >= %d
        LIMIT %d
      ", $offset, $cutoff, $cutoff, $batch));
      $msg = "Undid old featured image shifts (batch). Rows affected: {$affected}. Run again if needed.";
    }

    // FIX GALLERY IDS IN post_content (Classic + Gutenberg), batchwise by post ID
    if (isset($_POST['fix_galleries']) && self::nonce_ok('wpmf')) {
      $start_id = self::int('start_id', 0);

      $rows = $wpdb->get_results($wpdb->prepare("
        SELECT ID, post_content
        FROM {$wpdb->posts}
        WHERE ID > %d
          AND post_type IN ('post','page')
          AND post_status IN ('publish','draft','private','future')
          AND (
            post_content LIKE '%[gallery%'
            OR post_content LIKE '%wp:gallery%'
            OR post_content LIKE '%\"core/gallery\"%'
            OR post_content REGEXP '\"ids\"[[:space:]]*:[[:space:]]*\\[[0-9]'
          )
        ORDER BY ID ASC
        LIMIT %d
      ", $start_id, $batch));

      $updated = 0;
      $last_id = $start_id;

      foreach ($rows as $r) {
        $last_id = (int)$r->ID;
        $orig = (string)$r->post_content;
        $new  = $orig;

        // Classic: [gallery ids="1,2,3"]
        $new = preg_replace_callback(
          '/\\[gallery\\b[^\\]]*?\\bids="([^"]+)"[^\\]]*\\]/i',
          function($m) use ($offset) {
            $csv = $m[1];
            $parts = array_filter(array_map('trim', explode(',', $csv)), fn($v) => $v !== '');
            $out = [];
            foreach ($parts as $p) {
              $out[] = ctype_digit($p) ? (string)(intval($p) + $offset) : $p;
            }
            $bumped = implode(',', $out);
            return str_replace($csv, $bumped, $m[0]);
          },
          $new
        );

        // Gutenberg: "ids":[1,2,3]
        $new = preg_replace_callback(
          '/"ids"\\s*:\\s*\\[([0-9,\\s]+)\\]/',
          function($m) use ($offset) {
            $list = $m[1];
            $parts = array_filter(array_map('trim', explode(',', $list)), fn($v) => $v !== '');
            $out = [];
            foreach ($parts as $p) {
              $out[] = ctype_digit($p) ? (string)(intval($p) + $offset) : $p;
            }
            return '"ids":[' . implode(',', $out) . ']';
          },
          $new
        );

        if ($new !== $orig) {
          $wpdb->update($wpdb->posts, ['post_content' => $new], ['ID' => $r->ID]);
          $updated++;
        }
      }

      $msg = "Gallery fix batch done. Posts scanned: " . count($rows) . " | updated: {$updated} | last_id: {$last_id}. Use last_id as next start_id.";
    }

    // REGENERATE ATTACHMENT METADATA (batch)
    if (isset($_POST['regen_meta']) && self::nonce_ok('wpmf')) {
      $start_id = self::int('attach_start_id', 0);

      // Find attachments missing metadata first, then regenerate
      $atts = $wpdb->get_col($wpdb->prepare("
        SELECT p.ID
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} m
          ON m.post_id = p.ID AND m.meta_key = '_wp_attachment_metadata'
        WHERE p.post_type = 'attachment'
          AND p.ID > %d
          AND m.post_id IS NULL
        ORDER BY p.ID ASC
        LIMIT %d
      ", $start_id, $batch));

      $done = 0;
      $last = $start_id;

      require_once ABSPATH . 'wp-admin/includes/image.php';

      foreach ($atts as $aid) {
        $aid = (int)$aid;
        $last = $aid;

        $file = get_attached_file($aid);
        if (!$file || !file_exists($file)) continue;

        $meta = wp_generate_attachment_metadata($aid, $file);
        if (is_array($meta)) {
          wp_update_attachment_metadata($aid, $meta);
          $done++;
        }
      }

      $msg = "Regenerated attachment metadata (missing only). Batch processed: " . count($atts) . " | updated: {$done} | last_attachment_id: {$last}. Use last_attachment_id as next attach_start_id.";
    }

    // RECOUNT TERMS (categories + tags) - batchwise
    if (isset($_POST['recount_terms']) && self::nonce_ok('wpmf')) {

      $batch = max(50, intval($_POST['term_batch'] ?? 200));
      $taxes = ['category', 'post_tag'];

      $total = 0;
      foreach ($taxes as $tax) {
        $term_ids = get_terms([
          'taxonomy'   => $tax,
          'hide_empty' => false,
          'fields'     => 'ids',
          'number'     => $batch,
          'offset'     => intval($_POST['term_offset'] ?? 0),
        ]);

        if (!is_wp_error($term_ids) && !empty($term_ids)) {
          wp_update_term_count($term_ids, $tax);
          $total += count($term_ids);
        }
      }

      $msg = "Term recount batch done. Terms processed: {$total}. Increase term_offset and run again if needed.";
    }

    // UNDO GALLERY IDS FOR OLD POSTS (post_id < cutoff): subtract offset
if (isset($_POST['undo_galleries_old']) && self::nonce_ok('wpmf')) {
  $offset = intval($_POST['offset'] ?? 50000);
  $cutoff = intval($_POST['cutoff'] ?? 50000);
  $batch  = max(10, intval($_POST['batch'] ?? 200));
  $start_id = intval($_POST['undo_start_id'] ?? 0);

  $rows = $wpdb->get_results($wpdb->prepare("
    SELECT ID, post_content
    FROM {$wpdb->posts}
    WHERE ID > %d
      AND ID < %d
      AND post_type IN ('post','page')
      AND post_status IN ('publish','draft','private','future')
      AND (
        post_content LIKE '%[gallery%'
        OR post_content LIKE '%wp:gallery%'
        OR post_content REGEXP '\"ids\"[[:space:]]*:[[:space:]]*\\[[0-9]'
      )
    ORDER BY ID ASC
    LIMIT %d
  ", $start_id, $cutoff, $batch));

  $updated = 0;
  $last_id = $start_id;

  foreach ($rows as $r) {
    $last_id = (int)$r->ID;
    $orig = (string)$r->post_content;
    $new  = $orig;

    // Classic: [gallery ids="..."] -> subtract offset, but only if id >= offset
    $new = preg_replace_callback(
      '/\\[gallery\\b[^\\]]*?\\bids="([^"]+)"[^\\]]*\\]/i',
      function($m) use ($offset) {
        $csv = $m[1];
        $parts = array_filter(array_map('trim', explode(',', $csv)), fn($v) => $v !== '');
        $out = [];
        foreach ($parts as $p) {
          if (ctype_digit($p)) {
            $n = intval($p);
            // Only undo if it looks shifted
            $out[] = (string)(($n >= $offset) ? ($n - $offset) : $n);
          } else {
            $out[] = $p;
          }
        }
        $bumped = implode(',', $out);
        return str_replace($csv, $bumped, $m[0]);
      },
      $new
    );

    // Gutenberg: "ids":[...] -> subtract offset only if id >= offset
    $new = preg_replace_callback(
      '/"ids"\\s*:\\s*\\[([0-9,\\s]+)\\]/',
      function($m) use ($offset) {
        $list = $m[1];
        $parts = array_filter(array_map('trim', explode(',', $list)), fn($v) => $v !== '');
        $out = [];
        foreach ($parts as $p) {
          if (ctype_digit($p)) {
            $n = intval($p);
            $out[] = (string)(($n >= $offset) ? ($n - $offset) : $n);
          } else {
            $out[] = $p;
          }
        }
        return '"ids":[' . implode(',', $out) . ']';
      },
      $new
    );

    if ($new !== $orig) {
      $wpdb->update($wpdb->posts, ['post_content' => $new], ['ID' => $r->ID]);
      $updated++;
    }
  }

  $msg = "UNDO galleries (old posts) batch done. Scanned: " . count($rows) . " | updated: {$updated} | last_id: {$last_id}. Set undo_start_id=last_id and run again.";
}



    echo '<div class="wrap"><h1>WP Merge Fixer</h1>';

    if ($msg) {
      echo '<div class="notice notice-success"><p>' . esc_html($msg) . '</p></div>';
    }

    echo '<form method="post" style="max-width: 900px;">';
    wp_nonce_field('wpmf');

    echo '<table class="form-table"><tbody>';

    echo '<tr><th><label>Offset</label></th><td><input name="offset" type="number" value="' . esc_attr($offset) . '"> (z.B. 50000)</td></tr>';
    echo '<tr><th><label>Import-Cutoff (Post ID)</label></th><td><input name="cutoff" type="number" value="' . esc_attr($cutoff) . '"> (importierte Posts sind >= cutoff)</td></tr>';
    echo '<tr><th><label>Batch Size</label></th><td><input name="batch" type="number" value="' . esc_attr($batch) . '"> (200–500 empfohlen)</td></tr>';

    echo '</tbody></table>';

    echo '<p><button class="button button-secondary" name="dry_run" value="1">Dry-Run (Statistiken)</button></p>';

    if (is_array($stats)) {
      echo '<h2>Dry-Run Ergebnisse</h2><ul>';
      foreach ($stats as $k => $v) {
        echo '<li><strong>' . esc_html($k) . ':</strong> ' . esc_html((string)$v) . '</li>';
      }
      echo '</ul>';
    }

    echo '<hr><h2>1) Featured Images (_thumbnail_id)</h2>';
    echo '<p><button class="button button-primary" name="fix_thumb_imported" value="1">Fix imported featured images (+offset) (Batch)</button> ';
    echo '<button class="button" name="undo_thumb_old" value="1">Undo old posts featured images (-offset) (Batch)</button></p>';

    echo '<hr><h2>2) Galleries (Classic + Gutenberg + Jetpack)</h2>';
    echo '<p>Batchweise von <code>start_id</code> aus laufen lassen. Nach jedem Lauf den ausgegebenen <code>last_id</code> hier eintragen.</p>';
    echo '<p><label>start_id: </label><input name="start_id" type="number" value="' . esc_attr(self::int('start_id', 0)) . '"> ';
    echo '<button class="button button-primary" name="fix_galleries" value="1">Fix galleries in post_content (Batch)</button></p>';

        echo '<hr><h2>2-2) UNDO Galleries (nur alte Posts)</h2>';
echo '<p>Rückgängig machen für Posts < cutoff (z.B. 50000). Batchweise von undo_start_id aus.</p>';
echo '<p><label>undo_start_id: </label><input name="undo_start_id" type="number" value="' . esc_attr(intval($_POST['undo_start_id'] ?? 0)) . '"> ';
echo '<button class="button button-primary" name="undo_galleries_old" value="1">UNDO gallery IDs for old posts (Batch)</button></p>';


    echo '<hr><h2>3) Attachment Metadata regenerieren (fehlende nur)</h2>';
    echo '<p>Nur für Attachments, die <code>_wp_attachment_metadata</code> nicht haben. Batchweise (kann dauern).</p>';
    echo '<p><label>attach_start_id: </label><input name="attach_start_id" type="number" value="' . esc_attr(self::int('attach_start_id', 0)) . '"> ';
    echo '<button class="button button-primary" name="regen_meta" value="1">Regenerate missing attachment metadata (Batch)</button></p>';


    echo '<hr><h2>4) Term-Zähler neu berechnen (Categories & Tags)</h2>';
    echo '<p>Batchweise Neuberechnung der Beitrags-Zähler je Kategorie/Tag (ohne CLI).</p>';
    echo '<p><label>term_offset: </label><input name="term_offset" type="number" value="' . esc_attr(intval($_POST['term_offset'] ?? 0)) . '"> ';
    echo '<label>term_batch: </label><input name="term_batch" type="number" value="' . esc_attr(intval($_POST['term_batch'] ?? 200)) . '"> ';
    echo '<button class="button button-primary" name="recount_terms" value="1">Recount Terms (Batch)</button></p>';



    echo '</form></div>';
  }
}

WPMergeFixer::init();
